/*
 * Copyright © 2014 Intel Corporation
 * SPDX-License-Identifier: MIT
 */

#include <gtest/gtest.h>
#include "elk_vec4.h"

using namespace elk;

class copy_propagation_vec4_test : public ::testing::Test {
   virtual void SetUp();
   virtual void TearDown();

public:
   struct elk_compiler *compiler;
   struct elk_compile_params params;
   struct intel_device_info *devinfo;
   void *ctx;
   struct gl_shader_program *shader_prog;
   struct elk_vue_prog_data *prog_data;
   vec4_visitor *v;
};

class copy_propagation_vec4_visitor : public vec4_visitor
{
public:
   copy_propagation_vec4_visitor(struct elk_compiler *compiler,
                                 struct elk_compile_params *params,
                                 nir_shader *shader,
                                 struct elk_vue_prog_data *prog_data)
      : vec4_visitor(compiler, params, NULL, prog_data, shader,
                     false /* no_spills */, false)
   {
      prog_data->dispatch_mode = INTEL_DISPATCH_MODE_4X2_DUAL_OBJECT;
   }

protected:
   virtual dst_reg *make_reg_for_system_value(int /* location */)
   {
      UNREACHABLE("Not reached");
   }

   virtual void setup_payload()
   {
      UNREACHABLE("Not reached");
   }

   virtual void emit_prolog()
   {
      UNREACHABLE("Not reached");
   }

   virtual void emit_thread_end()
   {
      UNREACHABLE("Not reached");
   }

   virtual void emit_urb_write_header(int /* mrf */)
   {
      UNREACHABLE("Not reached");
   }

   virtual vec4_instruction *emit_urb_write_opcode(bool /* complete */)
   {
      UNREACHABLE("Not reached");
   }
};


void copy_propagation_vec4_test::SetUp()
{
   ctx = ralloc_context(NULL);
   compiler = rzalloc(ctx, struct elk_compiler);
   devinfo = rzalloc(ctx, struct intel_device_info);
   compiler->devinfo = devinfo;

   params = {};
   params.mem_ctx = ctx;

   prog_data = ralloc(ctx, struct elk_vue_prog_data);
   nir_shader *shader = nir_shader_create(ctx, MESA_SHADER_VERTEX, NULL);

   v = new copy_propagation_vec4_visitor(compiler, &params, shader, prog_data);

   devinfo->ver = 4;
   devinfo->verx10 = devinfo->ver * 10;
}

void copy_propagation_vec4_test::TearDown()
{
   delete v;
   v = NULL;

   ralloc_free(ctx);
   ctx = NULL;
}


static void
copy_propagation(vec4_visitor *v)
{
   const bool print = os_get_option("TEST_DEBUG");

   if (print) {
      fprintf(stderr, "instructions before:\n");
      v->dump_instructions();
   }

   v->calculate_cfg();
   v->opt_copy_propagation();

   if (print) {
      fprintf(stderr, "instructions after:\n");
      v->dump_instructions();
   }
}

TEST_F(copy_propagation_vec4_test, test_swizzle_swizzle)
{
   dst_reg a = dst_reg(v, glsl_vec4_type());
   dst_reg b = dst_reg(v, glsl_vec4_type());
   dst_reg c = dst_reg(v, glsl_vec4_type());

   v->emit(v->ADD(a, src_reg(a), src_reg(a)));

   v->emit(v->MOV(b, swizzle(src_reg(a), ELK_SWIZZLE4(ELK_SWIZZLE_Y,
                                                      ELK_SWIZZLE_Z,
                                                      ELK_SWIZZLE_W,
                                                      ELK_SWIZZLE_X))));

   vec4_instruction *test_mov =
      v->MOV(c, swizzle(src_reg(b), ELK_SWIZZLE4(ELK_SWIZZLE_Y,
                                                 ELK_SWIZZLE_Z,
                                                 ELK_SWIZZLE_W,
                                                 ELK_SWIZZLE_X)));
   v->emit(test_mov);

   copy_propagation(v);

   EXPECT_EQ(test_mov->src[0].nr, a.nr);
   EXPECT_EQ(test_mov->src[0].swizzle, ELK_SWIZZLE4(ELK_SWIZZLE_Z,
                                                    ELK_SWIZZLE_W,
                                                    ELK_SWIZZLE_X,
                                                    ELK_SWIZZLE_Y));
}

TEST_F(copy_propagation_vec4_test, test_swizzle_writemask)
{
   dst_reg a = dst_reg(v, glsl_vec4_type());
   dst_reg b = dst_reg(v, glsl_vec4_type());
   dst_reg c = dst_reg(v, glsl_vec4_type());

   v->emit(v->MOV(b, swizzle(src_reg(a), ELK_SWIZZLE4(ELK_SWIZZLE_X,
                                                      ELK_SWIZZLE_Y,
                                                      ELK_SWIZZLE_X,
                                                      ELK_SWIZZLE_Z))));

   v->emit(v->MOV(writemask(a, WRITEMASK_XYZ), elk_imm_f(1.0f)));

   vec4_instruction *test_mov =
      v->MOV(c, swizzle(src_reg(b), ELK_SWIZZLE4(ELK_SWIZZLE_W,
                                                 ELK_SWIZZLE_W,
                                                 ELK_SWIZZLE_W,
                                                 ELK_SWIZZLE_W)));
   v->emit(test_mov);

   copy_propagation(v);

   /* should not copy propagate */
   EXPECT_EQ(test_mov->src[0].nr, b.nr);
   EXPECT_EQ(test_mov->src[0].swizzle, ELK_SWIZZLE4(ELK_SWIZZLE_W,
                                                    ELK_SWIZZLE_W,
                                                    ELK_SWIZZLE_W,
                                                    ELK_SWIZZLE_W));
}
