/*
 * © Copyright 2017-2018 Alyssa Rosenzweig
 * SPDX-License-Identifier: MIT
 */

#ifndef __PAN_POOL_H__
#define __PAN_POOL_H__

#include <stddef.h>
#include <genxml/gen_macros.h>

#include "util/u_dynarray.h"

struct pan_ptr {
   /* CPU address */
   void *cpu;

   /* GPU address */
   uint64_t gpu;
};

static inline struct pan_ptr
pan_ptr_offset(struct pan_ptr ptr, int64_t offset)
{
   ptr.cpu = ((char *)ptr.cpu) + offset;
   ptr.gpu += offset;
   return ptr;
}

/* Represents grow-only memory. */

struct pan_pool {
   /* Minimum size for allocated BOs. */
   size_t slab_size;
};

static inline void
pan_pool_init(struct pan_pool *pool, size_t slab_size)
{
   pool->slab_size = slab_size;
}

/* Represents a fat pointer for GPU-mapped memory, returned from the transient
 * allocator and not used for much else */

struct pan_ptr pan_pool_alloc_aligned(struct pan_pool *pool, size_t sz,
                                      unsigned alignment);

#define PAN_POOL_ALLOCATOR(pool_subclass, alloc_func)                          \
   struct pan_ptr pan_pool_alloc_aligned(struct pan_pool *p, size_t sz,        \
                                         unsigned alignment)                   \
   {                                                                           \
      pool_subclass *pool = container_of(p, pool_subclass, base);              \
      return alloc_func(pool, sz, alignment);                                  \
   }

static inline uint64_t
pan_pool_upload_aligned(struct pan_pool *pool, const void *data, size_t sz,
                        unsigned alignment)
{
   struct pan_ptr transfer = pan_pool_alloc_aligned(pool, sz, alignment);

   if (transfer.cpu)
      memcpy(transfer.cpu, data, sz);

   return transfer.gpu;
}

static inline uint64_t
pan_pool_upload(struct pan_pool *pool, const void *data, size_t sz)
{
   return pan_pool_upload_aligned(pool, data, sz, sz);
}

struct pan_desc_alloc_info {
   unsigned size;
   unsigned align;
   unsigned nelems;
};

#define PAN_DESC_ARRAY(count, name)                                            \
   {                                                                           \
      .size = pan_size(name), .align = pan_alignment(name), .nelems = count,   \
   }

#define PAN_DESC(name) PAN_DESC_ARRAY(1, name)

#define PAN_DESC_AGGREGATE(...)                                                \
   (struct pan_desc_alloc_info[])                                              \
   {                                                                           \
      __VA_ARGS__, {0},                                                        \
   }

static inline struct pan_ptr
pan_pool_alloc_descs(struct pan_pool *pool,
                     const struct pan_desc_alloc_info *descs)
{
   unsigned size = 0;
   unsigned align = descs[0].align;

   for (unsigned i = 0; descs[i].size; i++) {
      assert(!(size & (descs[i].align - 1)));
      size += descs[i].size * descs[i].nelems;
   }

   return pan_pool_alloc_aligned(pool, size, align);
}

#define pan_pool_alloc_desc(pool, name)                                        \
   pan_pool_alloc_descs(pool, PAN_DESC_AGGREGATE(PAN_DESC(name)))

#define pan_pool_alloc_desc_array(pool, count, name)                           \
   pan_pool_alloc_descs(pool, PAN_DESC_AGGREGATE(PAN_DESC_ARRAY(count, name)))

#define pan_pool_alloc_desc_aggregate(pool, ...)                               \
   pan_pool_alloc_descs(pool, PAN_DESC_AGGREGATE(__VA_ARGS__))

#endif
